/*:
 * @target MZ
 * @plugindesc v1.0 変数から現在値/最大値を取得して、シンプルなゲージを画面に表示します（サイズ/色/縦横 変更可）
 * @author You
 *
 * @help
 * ■ 概要
 * ・ゲーム変数から「現在値」「最大値」を読み取り、ゲージのみを描画します。
 * ・サイズ/位置/色/縦横/枠線/Z順を指定できます。
 * ・マップ/バトル両方のシーンで表示可能。マップ切り替え後も維持します。
 * ・テキストは描きません（“ゲージだけ”）。
 *
 * ■ 使い方（例）
 * 1) プラグインを導入してON。
 * 2) プラグインコマンド「ゲージ表示(ShowGauge)」で作成。
 *    例）ID:1, X:40, Y:40, 幅:200, 高さ:16,
 *        現在変数:11, 最大変数:12, 色:#ff8,#f80, 背景:#222
 * 3) 以降、位置/色等を各コマンドで変更できます。
 * 4) 「非表示(HideGauge)」で削除します。
 *
 * ■ 注意
 * ・フェード/トーン/フラッシュの内側に入るよう、Spriteset の pictureContainer 配下に配置します。
 * ・そのためレイヤー/スプライトは Sprite_Clickable を用いています。
 *
 * @command ShowGauge
 * @text ゲージ表示
 * @arg id @type number @min 1 @text ID
 * @arg x @type number @text X
 * @arg y @type number @text Y
 * @arg width @type number @min 1 @text 幅
 * @arg height @type number @min 1 @text 高さ
 * @arg curVarId @type variable @text 現在値の変数
 * @arg maxVarId @type variable @text 最大値の変数
 * @arg color1 @type string @text 色1（左/下）
 * @arg color2 @type string @text 色2（右/上）
 * @arg backColor @type string @text 背景色
 * @arg frameColor @type string @text 枠線色
 * @arg frameWidth @type number @min 0 @text 枠線太さ(px)
 * @arg vertical @type boolean @text 縦ゲージ
 * @arg z @type number @text Z（同レイヤー内の並び順）
 *
 * @command MoveGauge
 * @text 移動
 * @arg id @type number @min 1 @text ID
 * @arg x @type number @text X
 * @arg y @type number @text Y
 *
 * @command ResizeGauge
 * @text サイズ変更
 * @arg id @type number @min 1 @text ID
 * @arg width @type number @min 1 @text 幅
 * @arg height @type number @min 1 @text 高さ
 *
 * @command RebindVariables
 * @text 変数の差し替え
 * @arg id @type number @min 1 @text ID
 * @arg curVarId @type variable @text 現在値の変数
 * @arg maxVarId @type variable @text 最大値の変数
 *
 * @command RecolorGauge
 * @text 色の変更
 * @arg id @type number @min 1 @text ID
 * @arg color1 @type string @text 色1（左/下）
 * @arg color2 @type string @text 色2（右/上）
 * @arg backColor @type string @text 背景色
 * @arg frameColor @type string @text 枠線色
 * @arg frameWidth @type number @min 0 @text 枠線太さ(px)
 *
 * @command SetVertical
 * @text 縦/横の切替
 * @arg id @type number @min 1 @text ID
 * @arg vertical @type boolean @text 縦ゲージ
 *
 * @command HideGauge
 * @text ゲージ非表示（削除）
 * @arg id @type number @min 1 @text ID
 *
 * @command ClearAllGauges
 * @text すべて削除
 */

(() => {
  "use strict";
  const PLUGIN_NAME = document.currentScript.src.match(/([^/]+)\.js$/)[1];

  // 永続ストア
  const sys = () => {
    if (!$gameSystem._varGauges) $gameSystem._varGauges = {};
    return $gameSystem._varGauges;
  };

  // ゲージスプライト（クリック可能ベース）
  class VariableGaugeSprite extends Sprite_Clickable {
    constructor(def) {
      super();
      this.bitmap = new Bitmap(def.width, def.height);
      this._def = { ...def };
      this.x = def.x;
      this.y = def.y;
      this.zIndex = def.z;
      this._lastCur = null;
      this._lastMax = null;
      this._lastRect = `${def.width}x${def.height}`;
      this._needsRedraw = true;
      this.redraw();
    }

    def() { return this._def; }

    update() {
      super.update();
      const cur = $gameVariables.value(this._def.curVarId) ?? 0;
      const max = $gameVariables.value(this._def.maxVarId) ?? 1;
      const rectKey = `${this._def.width}x${this._def.height}`;
      if (cur !== this._lastCur || max !== this._lastMax || rectKey !== this._lastRect || this._needsRedraw) {
        this._lastCur = cur;
        this._lastMax = max;
        this._lastRect = rectKey;
        this._needsRedraw = false;
        this.redraw();
      }
    }

    setPosition(x, y) {
      this.x = x; this.y = y;
      this._def.x = x; this._def.y = y;
    }

    setSize(w, h) {
      w = Math.max(1, Number(w)); h = Math.max(1, Number(h));
      this._def.width = w; this._def.height = h;
      this.bitmap = new Bitmap(w, h);
      this._needsRedraw = true;
    }

    setVariables(curVarId, maxVarId) {
      this._def.curVarId = Number(curVarId);
      this._def.maxVarId = Number(maxVarId);
      this._needsRedraw = true;
    }

    setColors(color1, color2, backColor, frameColor, frameWidth) {
      if (color1 != null) this._def.color1 = color1;
      if (color2 != null) this._def.color2 = color2;
      if (backColor != null) this._def.backColor = backColor;
      if (frameColor != null) this._def.frameColor = frameColor;
      if (frameWidth != null) this._def.frameWidth = Math.max(0, Number(frameWidth));
      this._needsRedraw = true;
    }

    setVertical(v) {
      this._def.vertical = !!v;
      this._needsRedraw = true;
    }

    redraw() {
      const w = this._def.width, h = this._def.height;
      const c1 = this._def.color1 ?? "#66ccff";
      const c2 = this._def.color2 ?? "#0066ff";
      const bg = this._def.backColor ?? "rgba(0,0,0,0.35)";
      const fcol = this._def.frameColor ?? "#000000";
      const fth = Math.max(0, Number(this._def.frameWidth ?? 1));
      const vertical = !!this._def.vertical;

      const bmp = this.bitmap;
      bmp.clear();

      // 背景
      bmp.fillRect(0, 0, w, h, bg);

      // 本体
      if (vertical) {
        const fillH = Math.round(h * this._rate());
        const y = h - fillH;
        if (fillH > 0) bmp.gradientFillRect(0, y, w, fillH, c1, c2, true);
      } else {
        const fillW = Math.round(w * this._rate());
        if (fillW > 0) bmp.gradientFillRect(0, 0, fillW, h, c1, c2, false);
      }

      // 枠線
      if (fth > 0) {
        for (let i = 0; i < fth; i++) {
          // MZ Bitmap には strokeRect が実装されています
          bmp.strokeRect(i, i, w - i * 2, h - i * 2, fcol);
        }
      }
    }

    _rate() {
      const cur = Math.max(0, Number($gameVariables.value(this._def.curVarId) ?? 0));
      const max = Math.max(1, Number($gameVariables.value(this._def.maxVarId) ?? 1));
      return Math.max(0, Math.min(1, cur / max));
    }
  }

  // 管理：ID→Sprite
  class VariableGaugeManager {
    static setupLayer(scene) {
      if (!scene._varGaugeLayer) {
        // ✅ Clickable にして pictureContainer 側の入力走査で落ちないようにする
        const layer = new Sprite_Clickable();
        layer.sortableChildren = true;
        scene._varGaugeLayer = layer;

        // Spriteset の pictureContainer 配下に置く（フェード/トーン/フラッシュの内側）
        const parent = (scene._spriteset && scene._spriteset._pictureContainer)
          ? scene._spriteset._pictureContainer
          : (scene._spriteset || null);

        if (parent) {
          parent.addChild(layer);
        } else if (scene._windowLayer) {
          // Fallback: WindowLayer の直下に差し込む
          const idx = Math.max(0, scene.children.indexOf(scene._windowLayer));
          scene.addChildAt(layer, idx);
        } else {
          // 最終保険
          scene.addChild(layer);
        }
      }
    }

    static rebuildForScene(scene) {
      this.setupLayer(scene);
      scene._varGaugeLayer.removeChildren();

      const store = sys();
      for (const idStr of Object.keys(store)) {
        const def = store[idStr];
        const sp = new VariableGaugeSprite(def);
        scene._varGaugeLayer.addChild(sp);
      }
      scene._varGaugeLayer.children.sort((a, b) => (a.zIndex ?? 0) - (b.zIndex ?? 0));
    }

    static spriteById(scene, id) {
      if (!scene || !scene._varGaugeLayer) return null;
      return scene._varGaugeLayer.children.find(s => s instanceof VariableGaugeSprite && s.def().id === id) || null;
    }
  }

  // シーン切替で復元
  const alias_SceneMap_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function () {
    alias_SceneMap_start.call(this);
    VariableGaugeManager.rebuildForScene(this);
  };

  const alias_SceneBattle_createDisplayObjects = Scene_Battle.prototype.createDisplayObjects;
  Scene_Battle.prototype.createDisplayObjects = function () {
    alias_SceneBattle_createDisplayObjects.call(this);
    VariableGaugeManager.rebuildForScene(this);
  };

  // コマンド登録
  PluginManager.registerCommand(PLUGIN_NAME, "ShowGauge", args => {
    const def = {
      id: Number(args.id),
      x: Number(args.x), y: Number(args.y),
      width: Number(args.width), height: Number(args.height),
      curVarId: Number(args.curVarId), maxVarId: Number(args.maxVarId),
      color1: args.color1, color2: args.color2, backColor: args.backColor,
      vertical: args.vertical === "true",
      z: Number(args.z ?? 10),
      frameColor: args.frameColor ?? "#000000",
      frameWidth: Number(args.frameWidth ?? 1)
    };
    const store = sys();
    store[def.id] = def;

    const scene = SceneManager._scene;
    if (scene && (scene instanceof Scene_Map || scene instanceof Scene_Battle)) {
      VariableGaugeManager.setupLayer(scene);
      const old = VariableGaugeManager.spriteById(scene, def.id);
      if (old) old.parent.removeChild(old);
      const sp = new VariableGaugeSprite(def);
      scene._varGaugeLayer.addChild(sp);
    }
  });

  PluginManager.registerCommand(PLUGIN_NAME, "MoveGauge", args => {
    const id = Number(args.id);
    const store = sys();
    if (!store[id]) return;
    store[id].x = Number(args.x);
    store[id].y = Number(args.y);

    const sp = VariableGaugeManager.spriteById(SceneManager._scene, id);
    if (sp) sp.setPosition(store[id].x, store[id].y);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "ResizeGauge", args => {
    const id = Number(args.id);
    const store = sys();
    if (!store[id]) return;
    store[id].width = Number(args.width);
    store[id].height = Number(args.height);

    const sp = VariableGaugeManager.spriteById(SceneManager._scene, id);
    if (sp) sp.setSize(store[id].width, store[id].height);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "RebindVariables", args => {
    const id = Number(args.id);
    const store = sys();
    if (!store[id]) return;
    store[id].curVarId = Number(args.curVarId);
    store[id].maxVarId = Number(args.maxVarId);

    const sp = VariableGaugeManager.spriteById(SceneManager._scene, id);
    if (sp) sp.setVariables(store[id].curVarId, store[id].maxVarId);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "RecolorGauge", args => {
    const id = Number(args.id);
    const store = sys();
    if (!store[id]) return;
    store[id].color1 = args.color1 ?? store[id].color1;
    store[id].color2 = args.color2 ?? store[id].color2;
    store[id].backColor = args.backColor ?? store[id].backColor;
    store[id].frameColor = args.frameColor ?? store[id].frameColor;
    store[id].frameWidth = Number(args.frameWidth ?? store[id].frameWidth ?? 1);

    const sp = VariableGaugeManager.spriteById(SceneManager._scene, id);
    if (sp) sp.setColors(store[id].color1, store[id].color2, store[id].backColor, store[id].frameColor, store[id].frameWidth);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "SetVertical", args => {
    const id = Number(args.id);
    const v = args.vertical === "true";
    const store = sys();
    if (!store[id]) return;
    store[id].vertical = v;

    const sp = VariableGaugeManager.spriteById(SceneManager._scene, id);
    if (sp) sp.setVertical(v);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "HideGauge", args => {
    const id = Number(args.id);
    const store = sys();
    if (!store[id]) return;
    delete store[id];

    const scene = SceneManager._scene;
    const sp = VariableGaugeManager.spriteById(scene, id);
    if (sp && sp.parent) sp.parent.removeChild(sp);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "ClearAllGauges", args => {
    $gameSystem._varGauges = {};
    const scene = SceneManager._scene;
    if (scene && scene._varGaugeLayer) scene._varGaugeLayer.removeChildren();
  });
})();


